@extends('backend.layouts.app')

@section('content')
<!-- Leaflet CSS -->
<link rel="stylesheet" href="https://unpkg.com/leaflet@1.9.4/dist/leaflet.css"
     integrity="sha256-p4NxAoJBhIIN+hmNHrzRCf9tD/miZyoHS5obTRR9BMY="
     crossorigin=""/>

<style>
    /* Form Controls Styling */
    .form-control {
        padding: 12px 16px;
        font-size: 16px;
        border-radius: 8px;
        border: 2px solid #e9ecef;
        transition: all 0.3s ease;
        font-weight: 500;
        background-color: #fff;
    }
    
    .form-control:focus {
        border-color: #0d6efd;
        box-shadow: 0 0 0 0.2rem rgba(13, 110, 253, 0.25);
        transform: translateY(-1px);
        background-color: #f8f9ff;
    }
    
    /* Enhanced Dropdown Styling */
    .form-select {
        padding: 12px 40px 12px 16px; /* Extra right padding for arrow */
        font-size: 16px;
        border-radius: 8px;
        border: 2px solid #e9ecef;
        transition: all 0.3s ease;
        font-weight: 500;
        background-color: #fff;
        cursor: pointer;
        /* Hide default arrow */
        -webkit-appearance: none;
        -moz-appearance: none;
        appearance: none;
        /* Add custom arrow */
        background-image: url("data:image/svg+xml,%3csvg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 16 16'%3e%3cpath fill='none' stroke='%23343a40' stroke-linecap='round' stroke-linejoin='round' stroke-width='2' d='m1 6 7 7-7'/%3e%3c/svg%3e");
        background-position: right 12px center;
        background-repeat: no-repeat;
        background-size: 16px;
    }
    
    .form-select:focus {
        border-color: #0d6efd;
        box-shadow: 0 0 0 0.2rem rgba(13, 110, 253, 0.25);
        transform: translateY(-1px);
        background-color: #f8f9ff;
        /* Change arrow color on focus */
        background-image: url("data:image/svg+xml,%3csvg xmlns='http://www.w3.org/2000/svg' viewBox='0 0 16 16'%3e%3cpath fill='none' stroke='%230d6efd' stroke-linecap='round' stroke-linejoin='round' stroke-width='2' d='m1 6 7 7-7'/%3e%3c/svg%3e");
    }
    
    /* Hover effects for form controls */
    .form-control:hover, .form-select:hover {
        border-color: #adb5bd;
        transform: translateY(-1px);
        box-shadow: 0 2px 8px rgba(0, 0, 0, 0.08);
    }
    
    /* Labels */
    .form-label {
        font-weight: 600;
        color: #2c3e50;
        margin-bottom: 8px;
        font-size: 15px;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }
    
    /* Cards */
    .card {
        border-radius: 12px;
        border: 1px solid #e9ecef;
        box-shadow: 0 4px 12px rgba(0, 0, 0, 0.05);
        transition: all 0.3s ease;
    }
    
    .card:hover {
        box-shadow: 0 8px 25px rgba(0, 0, 0, 0.1);
        transform: translateY(-2px);
    }
    
    .card-header {
        background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
        border-bottom: 2px solid #dee2e6;
        border-radius: 12px 12px 0 0 !important;
    }
    
    /* Buttons */
    .btn {
        border-radius: 8px;
        font-weight: 600;
        padding: 12px 24px;
        font-size: 16px;
        transition: all 0.3s ease;
        border: none;
        text-transform: uppercase;
        letter-spacing: 0.5px;
    }
    
    .btn-primary {
        background: linear-gradient(135deg, #0d6efd 0%, #009ef7 100%);
        box-shadow: 0 4px 12px rgba(13, 110, 253, 0.3);
    }
    
    .btn-primary:hover {
        background: linear-gradient(135deg, #0b5ed7 0%, #0089d6 100%);
        transform: translateY(-2px);
        box-shadow: 0 6px 20px rgba(13, 110, 253, 0.4);
    }
    
    .btn-secondary {
        background: linear-gradient(135deg, #6c757d 0%, #5a6268 100%);
        box-shadow: 0 4px 12px rgba(108, 117, 125, 0.3);
    }
    
    .btn-secondary:hover {
        background: linear-gradient(135deg, #5a6268 0%, #495057 100%);
        transform: translateY(-2px);
        box-shadow: 0 6px 20px rgba(108, 117, 125, 0.4);
    }
    
    /* Map styling */
    #map {
        height: 450px;
        width: 100%;
        border-radius: 8px;
        box-shadow: 0 2px 8px rgba(0,0,0,0.1);
    }
    
    .map-instructions {
        background: #e3f2fd;
        padding: 12px;
        border-radius: 6px;
        margin-bottom: 15px;
        border-left: 4px solid #2196F3;
    }
    
    .coordinates-display {
        background: #f5f5f5;
        padding: 10px;
        border-radius: 6px;
        font-family: monospace;
        font-size: 13px;
    }
    
    /* Input group styling */
    .input-group {
        box-shadow: 0 2px 5px rgba(0, 0, 0, 0.05);
        border-radius: 8px;
        overflow: hidden;
    }
    
    .input-group .form-control {
        border-radius: 0 8px 8px 0;
    }
    
    .input-group .btn {
        border-radius: 8px 0 0 8px;
    }
    
    /* Required field indicator */
    .text-danger {
        color: #dc3545;
        font-weight: 700;
    }
    
    /* Invalid feedback */
    .invalid-feedback {
        font-size: 0.875rem;
        color: #dc3545;
        margin-top: 0.25rem;
    }
    
    .is-invalid {
        border-color: #dc3545;
    }
    
    /* Responsive adjustments */
    @media (max-width: 768px) {
        .form-control, .form-select {
            padding: 10px 14px;
            font-size: 16px;
        }
        
        .btn {
            padding: 10px 20px;
            font-size: 15px;
        }
    }
</style>

<div class="container-fluid">
    <!-- Page Header -->
    <div class="row mb-4">
        <div class="col-12">
            <div class="d-flex justify-content-between align-items-center">
                <div>
                    <h2 class="mb-1">{{ isset($store) ? 'Edit Store' : 'Add New Store' }}</h2>
                    <p class="text-muted mb-0">{{ isset($store) ? 'Update store information' : 'Add a new store location' }}</p>
                </div>
                <a href="{{ route('admin.stores.index') }}" class="btn btn-secondary">
                    <i class="fas fa-arrow-left"></i> Back to List
                </a>
            </div>
        </div>
    </div>

    <form action="{{ isset($store) ? route('admin.stores.update', $store->id) : route('admin.stores.store') }}" 
          method="POST" 
          id="storeForm">
        @csrf
        @if(isset($store))
            @method('PUT')
        @endif

        <div class="row">
            <!-- Store Information -->
            <div class="col-lg-6">
                <div class="card mb-4">
                    <div class="card-header">
                        <h5 class="mb-0"><i class="fas fa-info-circle"></i> Store Information</h5>
                    </div>
                    <div class="card-body">
                        <!-- Store Name -->
                        <div class="mb-3">
                            <label for="name" class="form-label">Store Name <span class="text-danger">*</span></label>
                            <input type="text" 
                                   class="form-control @error('name') is-invalid @enderror" 
                                   id="name" 
                                   name="name" 
                                   value="{{ old('name', $store->name ?? '') }}" 
                                   required>
                            @error('name')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>

                        <!-- Address -->
                        <div class="mb-3">
                            <label for="address" class="form-label">Address <span class="text-danger">*</span></label>
                            <textarea class="form-control @error('address') is-invalid @enderror" 
                                      id="address" 
                                      name="address" 
                                      rows="3" 
                                      required>{{ old('address', $store->address ?? '') }}</textarea>
                            @error('address')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>

                        <!-- Location Information -->
                        <div class="mb-3">
                            <label for="division_id" class="form-label">Division <span class="text-danger">*</span></label>
                            <select class="form-select @error('division_id') is-invalid @enderror" 
                                    id="division_id" 
                                    name="division_id" 
                                    required>
                                <option value="">Select Division</option>
                                @foreach($divisions as $division)
                                    <option value="{{ $division->id }}" 
                                            {{ (isset($store) && $store->division_id == $division->id) || old('division_id') == $division->id ? 'selected' : '' }}>
                                        {{ $division->name_en }}
                                    </option>
                                @endforeach
                            </select>
                            @error('division_id')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>

                        <div class="mb-3">
                            <label for="district_id" class="form-label">District <span class="text-danger">*</span></label>
                            <select class="form-select @error('district_id') is-invalid @enderror" 
                                    id="district_id" 
                                    name="district_id" 
                                    required>
                                <option value="">Select District</option>
                                @if(isset($store) && isset($districts))
                                    @foreach($districts as $district)
                                        <option value="{{ $district->id }}" 
                                                {{ $store->district_id == $district->id ? 'selected' : '' }}>
                                            {{ $district->name_en }}
                                        </option>
                                    @endforeach
                                @endif
                            </select>
                            @error('district_id')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>

                        <div class="mb-3">
                            <label for="upazila_id" class="form-label">Upazila <span class="text-danger">*</span></label>
                            <select class="form-select @error('upazila_id') is-invalid @enderror" 
                                    id="upazila_id" 
                                    name="upazila_id" 
                                    required>
                                <option value="">Select Upazila</option>
                                @if(isset($store) && isset($upazilas))
                                    @foreach($upazilas as $upazila)
                                        <option value="{{ $upazila->id }}" 
                                                {{ $store->upazila_id == $upazila->id ? 'selected' : '' }}>
                                            {{ $upazila->name_en }}
                                        </option>
                                    @endforeach
                                @endif
                            </select>
                            @error('upazila_id')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>

                        <div class="mb-3">
                            <label for="union_id" class="form-label">Union</label>
                            <select class="form-select @error('union_id') is-invalid @enderror" 
                                    id="union_id" 
                                    name="union_id">
                                <option value="">Select Union</option>
                                @if(isset($store) && isset($unions))
                                    @foreach($unions as $union)
                                        <option value="{{ $union->id }}" 
                                                {{ $store->union_id == $union->id ? 'selected' : '' }}>
                                            {{ $union->name_en }}
                                        </option>
                                    @endforeach
                                @endif
                            </select>
                            @error('union_id')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>

                        <!-- City -->
                        <div class="mb-3">
                            <label for="city" class="form-label">City <span class="text-danger">*</span></label>
                            <input type="text" 
                                   class="form-control @error('city') is-invalid @enderror" 
                                   id="city" 
                                   name="city" 
                                   value="{{ old('city', $store->city ?? '') }}" 
                                   required>
                            @error('city')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>

                        <!-- Area -->
                        <div class="mb-3">
                            <label for="area" class="form-label">Area <span class="text-danger">*</span></label>
                            <input type="text" 
                                   class="form-control @error('area') is-invalid @enderror" 
                                   id="area" 
                                   name="area" 
                                   value="{{ old('area', $store->area ?? '') }}" 
                                   required>
                            @error('area')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>

                        <!-- Phone -->
                        <div class="mb-3">
                            <label for="phone" class="form-label">Phone <span class="text-danger">*</span></label>
                            <input type="text" 
                                   class="form-control @error('phone') is-invalid @enderror" 
                                   id="phone" 
                                   name="phone" 
                                   value="{{ old('phone', $store->phone ?? '') }}" 
                                   placeholder="+880 1711-123456"
                                   required>
                            @error('phone')
                                <div class="invalid-feedback">{{ $message }}</div>
                            @enderror
                        </div>
                    </div>
                </div>
            </div>

            <!-- Location Map -->
            <div class="col-lg-6">
                <div class="card mb-4">
                    <div class="card-header">
                        <h5 class="mb-0"><i class="fas fa-map-marker-alt"></i> Location Coordinates</h5>
                    </div>
                    <div class="card-body">
                        <div class="map-instructions">
                            <i class="fas fa-info-circle"></i> 
                            <strong>Click on the map</strong> to set the store's exact location. 
                            You can also search for an address or drag the marker.
                        </div>

                        <!-- Search Address -->
                        <div class="mb-3">
                            <label for="searchAddress" class="form-label">Search Location</label>
                            <div class="input-group">
                                <input type="text" 
                                       class="form-control" 
                                       id="searchAddress" 
                                       placeholder="Enter address to search...">
                                <button class="btn btn-primary" type="button" id="searchBtn">
                                    <i class="fas fa-search"></i> Search
                                </button>
                            </div>
                        </div>

                        <!-- Coordinates Display -->
                        <div class="row mb-3">
                            <div class="col-md-6">
                                <label for="latitude" class="form-label">Latitude <span class="text-danger">*</span></label>
                                <input type="text" 
                                       class="form-control @error('latitude') is-invalid @enderror" 
                                       id="latitude" 
                                       name="latitude" 
                                       value="{{ old('latitude', $store->latitude ?? '23.8103') }}" 
                                       readonly 
                                       required>
                                @error('latitude')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                            <div class="col-md-6">
                                <label for="longitude" class="form-label">Longitude <span class="text-danger">*</span></label>
                                <input type="text" 
                                       class="form-control @error('longitude') is-invalid @enderror" 
                                       id="longitude" 
                                       name="longitude" 
                                       value="{{ old('longitude', $store->longitude ?? '90.4125') }}" 
                                       readonly 
                                       required>
                                @error('longitude')
                                    <div class="invalid-feedback">{{ $message }}</div>
                                @enderror
                            </div>
                        </div>

                        <!-- Map -->
                        <div id="map"></div>

                        <div class="coordinates-display mt-3">
                            <strong>Selected Location:</strong>
                            <div id="coordinatesDisplay">Click on map to select location</div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Submit Buttons -->
        <div class="row">
            <div class="col-12">
                <div class="card">
                    <div class="card-body">
                        <div class="d-flex justify-content-between">
                            <a href="{{ route('admin.stores.index') }}" class="btn btn-secondary">
                                <i class="fas fa-times"></i> Cancel
                            </a>
                            <button type="submit" class="btn btn-primary">
                                <i class="fas fa-save"></i> {{ isset($store) ? 'Update Store' : 'Create Store' }}
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        </div>
    </form>
</div>

<!-- Leaflet JavaScript -->
<script src="https://unpkg.com/leaflet@1.9.4/dist/leaflet.js"
     integrity="sha256-20nQCchB9co0qIjJZRGuk2/Z9VM+kNiyxNV1lvTlZBo="
     crossorigin=""></script>

<script>
let map;
let marker;

// Initialize map
function initMap() {
    const lat = parseFloat(document.getElementById('latitude').value) || 23.8103;
    const lng = parseFloat(document.getElementById('longitude').value) || 90.4125;

    // Create map
    map = L.map('map').setView([lat, lng], 13);

    // Add tile layer
    L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
        attribution: '&copy; <a href="https://www.openstreetmap.org/copyright">OpenStreetMap</a> contributors',
        maxZoom: 19
    }).addTo(map);

    // Add marker
    marker = L.marker([lat, lng], {
        draggable: true
    }).addTo(map);

    // Update coordinates when marker is dragged
    marker.on('dragend', function(e) {
        const position = marker.getLatLng();
        updateCoordinates(position.lat, position.lng);
    });

    // Add marker on map click
    map.on('click', function(e) {
        marker.setLatLng(e.latlng);
        updateCoordinates(e.latlng.lat, e.latlng.lng);
    });

    // Update display
    updateCoordinatesDisplay(lat, lng);
}

// Update coordinates
function updateCoordinates(lat, lng) {
    document.getElementById('latitude').value = lat.toFixed(7);
    document.getElementById('longitude').value = lng.toFixed(7);
    updateCoordinatesDisplay(lat, lng);
}

// Update coordinates display
function updateCoordinatesDisplay(lat, lng) {
    document.getElementById('coordinatesDisplay').innerHTML = 
        `Latitude: <strong>${lat.toFixed(7)}</strong>, Longitude: <strong>${lng.toFixed(7)}</strong>`;
}

// Search address using Nominatim (OpenStreetMap)
document.getElementById('searchBtn').addEventListener('click', async function() {
    const address = document.getElementById('searchAddress').value;
    
    if (!address) {
        alert('Please enter an address to search');
        return;
    }

    try {
        const response = await fetch(`https://nominatim.openstreetmap.org/search?format=json&q=${encodeURIComponent(address)}&countrycodes=bd&limit=1`);
        const data = await response.json();

        if (data && data.length > 0) {
            const lat = parseFloat(data[0].lat);
            const lng = parseFloat(data[0].lon);

            // Update map
            map.setView([lat, lng], 16);
            marker.setLatLng([lat, lng]);
            updateCoordinates(lat, lng);
        } else {
            alert('Location not found. Please try a different search term.');
        }
    } catch (error) {
        console.error('Search error:', error);
        alert('Error searching for location. Please try again.');
    }
});

// Handle division change
document.getElementById('division_id').addEventListener('change', function() {
    const divisionId = this.value;
    const districtSelect = document.getElementById('district_id');
    
    if (divisionId) {
        fetch(`/admin/stores/get-districts/${divisionId}`)
            .then(response => response.json())
            .then(data => {
                districtSelect.innerHTML = '<option value="">Select District</option>';
                if (data.success && data.data) {
                    data.data.forEach(district => {
                        districtSelect.innerHTML += `<option value="${district.id}">${district.name_en}</option>`;
                    });
                }
            })
            .catch(error => console.error('Error:', error));
    } else {
        districtSelect.innerHTML = '<option value="">Select District</option>';
    }
    
    // Reset dependent selects
    document.getElementById('upazila_id').innerHTML = '<option value="">Select Upazila</option>';
    document.getElementById('union_id').innerHTML = '<option value="">Select Union</option>';
});

// Handle district change
document.getElementById('district_id').addEventListener('change', function() {
    const districtId = this.value;
    const upazilaSelect = document.getElementById('upazila_id');
    
    if (districtId) {
        fetch(`/admin/stores/get-upazilas/${districtId}`)
            .then(response => response.json())
            .then(data => {
                upazilaSelect.innerHTML = '<option value="">Select Upazila</option>';
                if (data.success && data.data) {
                    data.data.forEach(upazila => {
                        upazilaSelect.innerHTML += `<option value="${upazila.id}">${upazila.name_en}</option>`;
                    });
                }
            })
            .catch(error => console.error('Error:', error));
    } else {
        upazilaSelect.innerHTML = '<option value="">Select Upazila</option>';
    }
    
    // Reset dependent select
    document.getElementById('union_id').innerHTML = '<option value="">Select Union</option>';
});

// Handle upazila change
document.getElementById('upazila_id').addEventListener('change', function() {
    const upazilaId = this.value;
    const unionSelect = document.getElementById('union_id');
    
    if (upazilaId) {
        fetch(`/admin/stores/get-unions/${upazilaId}`)
            .then(response => response.json())
            .then(data => {
                unionSelect.innerHTML = '<option value="">Select Union</option>';
                if (data.success && data.data) {
                    data.data.forEach(union => {
                        unionSelect.innerHTML += `<option value="${union.id}">${union.name_en}</option>`;
                    });
                }
            })
            .catch(error => console.error('Error:', error));
    } else {
        unionSelect.innerHTML = '<option value="">Select Union</option>';
    }
});

// Initialize map when page loads
document.addEventListener('DOMContentLoaded', function() {
    initMap();
});

// Form validation
document.getElementById('storeForm').addEventListener('submit', function(e) {
    const lat = document.getElementById('latitude').value;
    const lng = document.getElementById('longitude').value;

    if (!lat || !lng) {
        e.preventDefault();
        alert('Please select a location on the map');
        return false;
    }
});
</script>

@endsection