/**
 * Store Locator - Leaflet Integration with Pagination & Card View
 * Handles map initialization, store loading, pagination, and interactive cards
 */

let map;
let markers = [];
let markerGroup;
let cities = [];
let areas = [];
let allStores = [];
let currentPage = 1;
let itemsPerPage = 6;
let activeStoreId = null;

/**
 * Initialize Leaflet Map
 * Called automatically when page loads
 */
function initMap() {
    // Center map on Bangladesh
    const bangladeshCenter = [23.8103, 90.4125];

    // Create map instance
    map = L.map('storeMap').setView(bangladeshCenter, 7);

    // Add OpenStreetMap tile layer
    L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png', {
        attribution: '&copy; <a href="https://www.openstreetmap.org/copyright">OpenStreetMap</a> contributors',
        maxZoom: 19
    }).addTo(map);

    // Create layer group for markers
    markerGroup = L.layerGroup().addTo(map);

    // Load cities for dropdown
    loadCities();

    // Load all stores on initial page load
    loadStores();

    // Attach event listeners
    setupEventListeners();
}

/**
 * Setup all event listeners
 */
function setupEventListeners() {
    const citySelect = document.getElementById('citySelect');
    const areaSelect = document.getElementById('areaSelect');
    const resetBtn = document.getElementById('resetBtn');

    // City selection change - load areas and update map
    citySelect.addEventListener('change', function() {
        const selectedCity = this.value;
        
        if (selectedCity) {
            loadAreas(selectedCity);
            areaSelect.disabled = false;
            // Auto-update stores when city changes
            loadStores(selectedCity, '');
        } else {
            areaSelect.innerHTML = '<option value="">Select Area</option>';
            areaSelect.disabled = true;
            // Show all stores when city is cleared
            loadStores();
        }
        
        // Reset to first page
        currentPage = 1;
    });

    // Area selection change - auto-update map
    areaSelect.addEventListener('change', function() {
        const city = citySelect.value;
        const area = this.value;
        loadStores(city, area);
        
        // Reset to first page
        currentPage = 1;
    });

    // Reset button click
    resetBtn.addEventListener('click', function() {
        citySelect.value = '';
        areaSelect.innerHTML = '<option value="">Select Area</option>';
        areaSelect.disabled = true;
        currentPage = 1;
        activeStoreId = null;
        loadStores();
    });
}

/**
 * Load all available cities
 */
async function loadCities() {
    try {
        const response = await fetch('/api/v2/stores/cities', {
            method: 'GET',
            headers: {
                'Accept': 'application/json',
                'Content-Type': 'application/json',
            }
        });

        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }

        const result = await response.json();

        if (result.success && result.data) {
            cities = result.data;
            populateCityDropdown(cities);
        }
    } catch (error) {
        console.error('Error loading cities:', error);
    }
}

/**
 * Populate city dropdown with options
 * @param {Array} cityList - Array of city names
 */
function populateCityDropdown(cityList) {
    const citySelect = document.getElementById('citySelect');
    citySelect.innerHTML = '<option value="">All Cities</option>';
    
    cityList.forEach(city => {
        const option = document.createElement('option');
        option.value = city;
        option.textContent = city;
        citySelect.appendChild(option);
    });
}

/**
 * Load areas for a specific city
 * @param {string} city - City name
 */
async function loadAreas(city) {
    try {
        const response = await fetch(`/api/v2/stores/areas?city=${encodeURIComponent(city)}`, {
            method: 'GET',
            headers: {
                'Accept': 'application/json',
                'Content-Type': 'application/json',
            }
        });

        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }

        const result = await response.json();

        if (result.success && result.data) {
            areas = result.data;
            populateAreaDropdown(areas);
        }
    } catch (error) {
        console.error('Error loading areas:', error);
    }
}

/**
 * Populate area dropdown with options
 * @param {Array} areaList - Array of area names
 */
function populateAreaDropdown(areaList) {
    const areaSelect = document.getElementById('areaSelect');
    areaSelect.innerHTML = '<option value="">All Areas</option>';
    
    areaList.forEach(area => {
        const option = document.createElement('option');
        option.value = area;
        option.textContent = area;
        areaSelect.appendChild(option);
    });
}

/**
 * Load stores from API and display on map
 * @param {string} city - Optional city filter
 * @param {string} area - Optional area filter
 */
async function loadStores(city = '', area = '') {
    try {
        // Show loading indicator
        showLoading(true);

        // Build API URL with query parameters
        let apiUrl = '/api/v2/stores';
        const params = new URLSearchParams();
        
        if (city) params.append('city', city);
        if (area) params.append('area', area);
        
        if (params.toString()) {
            apiUrl += '?' + params.toString();
        }

        // Fetch stores from API
        const response = await fetch(apiUrl, {
            method: 'GET',
            headers: {
                'Accept': 'application/json',
                'Content-Type': 'application/json',
            }
        });

        if (!response.ok) {
            throw new Error(`HTTP error! status: ${response.status}`);
        }

        const result = await response.json();

        if (result.success && result.data) {
            allStores = result.data;
            
            // Clear existing markers
            clearMarkers();

            // Display all stores on map
            displayStoresOnMap(allStores);

            // Display paginated stores in cards
            displayStoreCards();

            // Update result count
            updateResultCount(result.count);

            console.log(`Found ${result.count} store(s)`);
        }
    } catch (error) {
        console.error('Error loading stores:', error);
        showError('Failed to load stores. Please try again.');
    } finally {
        showLoading(false);
    }
}

/**
 * Display stores as markers on the map
 * @param {Array} stores - Array of store objects
 */
function displayStoresOnMap(stores) {
    if (!stores || stores.length === 0) {
        // Reset to Bangladesh center
        map.setView([23.8103, 90.4125], 7);
        return;
    }

    const bounds = [];

    stores.forEach((store, index) => {
        const position = [parseFloat(store.latitude), parseFloat(store.longitude)];

        // Create custom icon
        const customIcon = L.icon({
            iconUrl: '/assets/img/pin.png',
            iconSize: [32, 32],
            iconAnchor: [16, 32],
            popupAnchor: [0, -32]
        });

        // Create marker
        const marker = L.marker(position, {
            icon: customIcon,
            title: store.name,
            storeId: index
        });

        // Create popup content
        const popupContent = createPopupContent(store);

        // Bind popup to marker
        marker.bindPopup(popupContent);

        // Add click listener to marker
        marker.on('click', function() {
            highlightStoreCard(index);
        });

        // Add marker to layer group
        marker.addTo(markerGroup);

        // Store marker reference
        markers.push(marker);

        // Add position to bounds array
        bounds.push(position);
    });

    // Fit map to show all markers
    if (bounds.length > 0) {
        map.fitBounds(bounds, {
            padding: [50, 50],
            maxZoom: stores.length === 1 ? 16 : 18
        });
    }
}

/**
 * Create popup content HTML
 * @param {Object} store - Store object
 * @returns {string} HTML content
 */
function createPopupContent(store) {
    return `
        <div style="padding: 10px; max-width: 250px;">
            <h3 style="margin: 0 0 10px 0; color: #333; font-size: 16px;">${store.name}</h3>
            <p style="margin: 5px 0; color: #666; font-size: 14px;">
                <strong>Address:</strong><br>
                ${store.address}<br>
                ${store.area}, ${store.city}
            </p>
            ${store.phone ? `<p style="margin: 5px 0; color: #666; font-size: 14px;">
                <strong>Phone:</strong> ${store.phone}
            </p>` : ''}
        </div>
    `;
}

/**
 * Display store cards with pagination
 */
function displayStoreCards() {
    const storeListEl = document.getElementById('storeList');
    
    if (!allStores || allStores.length === 0) {
        storeListEl.innerHTML = `
            <div class="empty-state">
                <div class="empty-state-icon">📍</div>
                <div class="empty-state-text">No stores found</div>
            </div>
        `;
        updatePagination(0);
        return;
    }

    // Calculate pagination
    const totalPages = Math.ceil(allStores.length / itemsPerPage);
    const startIndex = (currentPage - 1) * itemsPerPage;
    const endIndex = startIndex + itemsPerPage;
    const paginatedStores = allStores.slice(startIndex, endIndex);

    // Generate cards HTML
    let cardsHTML = '';
    paginatedStores.forEach((store, index) => {
        const globalIndex = startIndex + index;
        cardsHTML += createStoreCard(store, globalIndex);
    });

    storeListEl.innerHTML = cardsHTML;

    // Add click listeners to cards
    document.querySelectorAll('.store-card').forEach((card, index) => {
        const globalIndex = startIndex + index;
        card.addEventListener('click', () => focusOnStore(globalIndex));
    });

    // Update pagination controls
    updatePagination(allStores.length);
}

/**
 * Create store card HTML
 * @param {Object} store - Store object
 * @param {number} index - Store index
 * @returns {string} HTML content
 */
function createStoreCard(store, index) {
    const isActive = activeStoreId === index ? 'active' : '';
    return `
        <div class="store-card ${isActive}" data-store-id="${index}">
            <div class="store-card-header">
                <h3 class="store-name">${store.name}</h3>
                <span class="store-badge">${store.city}</span>
            </div>
            <div class="store-location">
                <span class="store-location-icon">📍</span>
                <div>
                    <div>${store.address}</div>
                    <div>${store.area}, ${store.city}</div>
                </div>
            </div>
            ${store.phone ? `
                <div class="store-phone">
                    <span class="store-phone-icon">📞</span>
                    <span>${store.phone}</span>
                </div>
            ` : ''}
            <button class="store-view-map">📍 View on Map</button>
        </div>
    `;
}

/**
 * Focus on a specific store on the map
 * @param {number} index - Store index
 */
function focusOnStore(index) {
    if (!allStores[index]) return;

    const store = allStores[index];
    const position = [parseFloat(store.latitude), parseFloat(store.longitude)];

    // Set active store
    activeStoreId = index;

    // Center map on store
    map.setView(position, 16, {
        animate: true,
        duration: 0.5
    });

    // Open marker popup
    if (markers[index]) {
        markers[index].openPopup();
    }

    // Highlight the card
    highlightStoreCard(index);
}

/**
 * Highlight a store card
 * @param {number} index - Store index
 */
function highlightStoreCard(index) {
    // Remove active class from all cards
    document.querySelectorAll('.store-card').forEach(card => {
        card.classList.remove('active');
    });

    // Add active class to clicked card
    const card = document.querySelector(`.store-card[data-store-id="${index}"]`);
    if (card) {
        card.classList.add('active');
        card.scrollIntoView({ behavior: 'smooth', block: 'nearest' });
    }

    activeStoreId = index;
}

/**
 * Update pagination controls
 * @param {number} totalItems - Total number of items
 */
function updatePagination(totalItems) {
    const totalPages = Math.ceil(totalItems / itemsPerPage);
    const paginationInfo = document.getElementById('paginationInfo');
    const paginationControls = document.getElementById('paginationControls');

    if (totalItems === 0) {
        paginationInfo.textContent = 'No stores';
        paginationControls.innerHTML = '';
        return;
    }

    // Update info
    const startIndex = (currentPage - 1) * itemsPerPage + 1;
    const endIndex = Math.min(currentPage * itemsPerPage, totalItems);
    paginationInfo.textContent = `Showing ${startIndex}-${endIndex} of ${totalItems}`;

    // Generate pagination buttons
    let buttonsHTML = '';
    
    // Previous button
    buttonsHTML += `
        <button class="pagination-btn" id="prevBtn" ${currentPage === 1 ? 'disabled' : ''}>
            ← Prev
        </button>
    `;

    // Page numbers
    for (let i = 1; i <= totalPages; i++) {
        if (
            i === 1 || 
            i === totalPages || 
            (i >= currentPage - 1 && i <= currentPage + 1)
        ) {
            buttonsHTML += `
                <button class="pagination-btn ${i === currentPage ? 'active' : ''}" data-page="${i}">
                    ${i}
                </button>
            `;
        } else if (i === currentPage - 2 || i === currentPage + 2) {
            buttonsHTML += `<span style="padding: 8px;">...</span>`;
        }
    }

    // Next button
    buttonsHTML += `
        <button class="pagination-btn" id="nextBtn" ${currentPage === totalPages ? 'disabled' : ''}>
            Next →
        </button>
    `;

    paginationControls.innerHTML = buttonsHTML;

    // Add click listeners
    document.getElementById('prevBtn')?.addEventListener('click', () => {
        if (currentPage > 1) {
            currentPage--;
            displayStoreCards();
        }
    });

    document.getElementById('nextBtn')?.addEventListener('click', () => {
        if (currentPage < totalPages) {
            currentPage++;
            displayStoreCards();
        }
    });

    document.querySelectorAll('.pagination-btn[data-page]').forEach(btn => {
        btn.addEventListener('click', (e) => {
            currentPage = parseInt(e.target.dataset.page);
            displayStoreCards();
        });
    });
}

/**
 * Update result count display
 * @param {number} count - Number of stores found
 */
function updateResultCount(count) {
    const resultCountEl = document.getElementById('resultCount');
    if (resultCountEl) {
        if (count === 0) {
            resultCountEl.textContent = 'No stores found';
            resultCountEl.style.color = '#999';
        } else {
            resultCountEl.textContent = `${count} store${count !== 1 ? 's' : ''} found`;
            resultCountEl.style.color = '#4285f4';
        }
    }
}

/**
 * Show/hide loading indicator
 * @param {boolean} show - Whether to show loading indicator
 */
function showLoading(show) {
    const storeListEl = document.getElementById('storeList');
    const resultCountEl = document.getElementById('resultCount');
    
    if (show) {
        storeListEl.innerHTML = `
            <div class="loading-state">
                <div class="spinner"></div>
                <p>Loading stores...</p>
            </div>
        `;
        resultCountEl.textContent = 'Loading...';
        resultCountEl.style.color = '#999';
    }
}

/**
 * Show error message
 * @param {string} message - Error message
 */
function showError(message) {
    const storeListEl = document.getElementById('storeList');
    storeListEl.innerHTML = `
        <div class="empty-state">
            <div class="empty-state-icon">⚠️</div>
            <div class="empty-state-text">${message}</div>
        </div>
    `;
}


// Handle division change
document.getElementById('division_id').addEventListener('change', function() {
    const divisionId = this.value;
    const districtSelect = document.getElementById('district_id');
    
    if (divisionId) {
        fetch(`/stores/get-districts/${divisionId}`)
            .then(response => response.json())
            .then(data => {
                districtSelect.innerHTML = '<option value="">Select District</option>';
                if (data.success && data.data) {
                    data.data.forEach(district => {
                        districtSelect.innerHTML += `<option value="${district.id}">${district.name_en}</option>`;
                    });
                }
            })
            .catch(error => console.error('Error:', error));
    } else {
        districtSelect.innerHTML = '<option value="">Select District</option>';
    }
    
    // Reset dependent selects
    document.getElementById('upazila_id').innerHTML = '<option value="">Select Upazila</option>';
    document.getElementById('union_id').innerHTML = '<option value="">Select Union</option>';
});

// Handle district change
document.getElementById('district_id').addEventListener('change', function() {
    const districtId = this.value;
    const upazilaSelect = document.getElementById('upazila_id');
    
    if (districtId) {
        fetch(`/stores/get-upazilas/${districtId}`)
            .then(response => response.json())
            .then(data => {
                upazilaSelect.innerHTML = '<option value="">Select Upazila</option>';
                if (data.success && data.data) {
                    data.data.forEach(upazila => {
                        upazilaSelect.innerHTML += `<option value="${upazila.id}">${upazila.name_en}</option>`;
                    });
                }
            })
            .catch(error => console.error('Error:', error));
    } else {
        upazilaSelect.innerHTML = '<option value="">Select Upazila</option>';
    }
    
    // Reset dependent select
    document.getElementById('union_id').innerHTML = '<option value="">Select Union</option>';
});

// Handle upazila change
document.getElementById('upazila_id').addEventListener('change', function() {
    const upazilaId = this.value;
    const unionSelect = document.getElementById('union_id');
    
    if (upazilaId) {
        fetch(`/stores/get-unions/${upazilaId}`)
            .then(response => response.json())
            .then(data => {
                unionSelect.innerHTML = '<option value="">Select Union</option>';
                if (data.success && data.data) {
                    data.data.forEach(union => {
                        unionSelect.innerHTML += `<option value="${union.id}">${union.name_en}</option>`;
                    });
                }
            })
            .catch(error => console.error('Error:', error));
    } else {
        unionSelect.innerHTML = '<option value="">Select Union</option>';
    }
});

/**
 * Clear all markers from the map
 */
function clearMarkers() {
    markerGroup.clearLayers();
    markers = [];
}

// Initialize map when DOM is ready
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', initMap);
} else {
    initMap();
}