<?php

use Illuminate\Database\Seeder;
use App\Models\Division;
use App\Models\District;
use App\Models\Upazila;
use App\Models\Union;

class AdministrativeDataSeeder extends Seeder
{
    /**
     * Run the database seeds.
     */
    public function run(): void
    {
        // Load English JSON
        $enPath = database_path('seeders/data/administrative_en.json');
        
        if (!file_exists($enPath)) {
            $this->command->error("Error: English JSON file not found at: " . $enPath);
            return;
        }
        
        $data = json_decode(file_get_contents($enPath), true);

        if (!isset($data['divisions'])) {
            $this->command->error("Error: Invalid JSON structure. 'divisions' key not found.");
            return;
        }

        $this->command->info("Starting to seed administrative data...");

        // Process divisions
        foreach ($data['divisions'] as $divisionData) {
            $this->command->info("Processing Division: " . $divisionData['name']);

            $division = Division::create([
                'name_bn' => $divisionData['name'], // Using English name for both if Bangla not available
                'name_en' => $divisionData['name'],
            ]);

            // Process districts
            if (isset($divisionData['districts']) && is_array($divisionData['districts'])) {
                foreach ($divisionData['districts'] as $districtData) {
                    $this->command->info("  - Processing District: " . $districtData['name']);

                    $district = District::create([
                        'division_id' => $division->id,
                        'name_bn' => $districtData['name'],
                        'name_en' => $districtData['name'],
                    ]);

                    // Process upazilas
                    if (isset($districtData['upazilas']) && is_array($districtData['upazilas'])) {
                        foreach ($districtData['upazilas'] as $upazilaData) {
                            // Check if upazila has unions or is just a string
                            $upazilaName = is_array($upazilaData) ? $upazilaData['name'] : $upazilaData;

                            $upazila = Upazila::create([
                                'district_id' => $district->id,
                                'name_bn' => $upazilaName,
                                'name_en' => $upazilaName,
                            ]);

                            // Process unions if they exist
                            if (is_array($upazilaData) && isset($upazilaData['unions'])) {
                                foreach ($upazilaData['unions'] as $unionName) {
                                    Union::create([
                                        'upazila_id' => $upazila->id,
                                        'name_bn' => $unionName,
                                        'name_en' => $unionName,
                                    ]);
                                }
                            }
                        }
                    }
                }
            }
        }

        $this->command->info("Administrative data seeding completed successfully!");
    }
}