<?php


namespace App\Services;

use Illuminate\Support\Facades\Http;

class LocationService
{
    const EARTH_RADIUS_KM = 6371;

    /**
     * Calculate distance between two coordinates.
     */
    public function calculateDistance($lat1, $lng1, $lat2, $lng2, $unit = 'km')
    {
        $latFrom = deg2rad($lat1);
        $lonFrom = deg2rad($lng1);
        $latTo   = deg2rad($lat2);
        $lonTo   = deg2rad($lng2);

        $latDelta = $latTo - $latFrom;
        $lonDelta = $lonTo - $lonFrom;

        $angle = 2 * asin(sqrt(pow(sin($latDelta / 2), 2) +
            cos($latFrom) * cos($latTo) * pow(sin($lonDelta / 2), 2)));

        $distance = self::EARTH_RADIUS_KM * $angle;

        return $unit === 'miles' ? $distance * 0.621371 : $distance;
    }

    /**
     * Convert address to latitude and longitude using Nominatim API.
     */
    public function geocodeAddress(string $address): ?array
    {
        $encodedAddress = urlencode($address);

        $url = "https://nominatim.openstreetmap.org/search?q={$encodedAddress}&format=json&limit=1";

        $response = Http::withHeaders([
            'User-Agent' => 'MyLaravelApp/1.0 (hello@example.com)' // This is required by Nominatim's usage policy
        ])->get($url);

        $results = $response->json();

        if ($response->successful() && !empty($results)) {
            return [
                'lat' => (float) $results[0]['lat'],
                'lng' => (float) $results[0]['lon'],
            ];
        }

        return null; // or throw an exception/log error if needed
    }
}
