<?php
namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;

class Store extends Model
{
    use HasFactory;

    protected $fillable = [
        'name',
        'address',
        'division_id',
        'district_id',
        'upazila_id',
        'union_id',
        'city',
        'area',
        'latitude',
        'longitude',
        'phone',
    ];

    protected $casts = [
        'latitude' => 'decimal:7',
        'longitude' => 'decimal:7',
    ];

    // ============================================
    // RELATIONSHIPS
    // ============================================
    
    public function division()
    {
        return $this->belongsTo(Division::class);
    }

    public function district()
    {
        return $this->belongsTo(District::class);
    }

    public function upazila()
    {
        return $this->belongsTo(Upazila::class);
    }

    public function union()
    {
        return $this->belongsTo(Union::class);
    }

    // ============================================
    // SCOPES
    // ============================================
    
    public function scopeByDivision($query, $divisionId)
    {
        if ($divisionId) {
            return $query->where('division_id', $divisionId);
        }
        return $query;
    }

    public function scopeByDistrict($query, $districtId)
    {
        if ($districtId) {
            return $query->where('district_id', $districtId);
        }
        return $query;
    }

    public function scopeByUpazila($query, $upazilaId)
    {
        if ($upazilaId) {
            return $query->where('upazila_id', $upazilaId);
        }
        return $query;
    }

    public function scopeByUnion($query, $unionId)
    {
        if ($unionId) {
            return $query->where('union_id', $unionId);
        }
        return $query;
    }

    public function scopeByCity($query, $city)
    {
        if ($city) {
            return $query->where('city', 'like', '%' . $city . '%');
        }
        return $query;
    }

    public function scopeByArea($query, $area)
    {
        if ($area) {
            return $query->where('area', 'like', '%' . $area . '%');
        }
        return $query;
    }

    // ============================================
    // ACCESSOR: Get Full Address
    // ============================================
    
    public function getFullAddressAttribute()
    {
        $parts = array_filter([
            $this->address,
            $this->union?->name_en,
            $this->upazila?->name_en,
            $this->district?->name_en,
            $this->division?->name_en,
        ]);
        
        return implode(', ', $parts);
    }

    public function getFullAddressBnAttribute()
    {
        $parts = array_filter([
            $this->address,
            $this->union?->name_bn,
            $this->upazila?->name_bn,
            $this->district?->name_bn,
            $this->division?->name_bn,
        ]);
        
        return implode(', ', $parts);
    }
}
