<?php

namespace App\Http\Controllers;

use App\Http\Controllers\Controller;
use App\Models\Store;
use App\Models\Division;
use App\Models\District;
use App\Models\Upazila;
use App\Models\Union;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;

class StoreAdminController extends Controller
{

    /**
     * Display a listing of the stores.
     */
    public function index(Request $request)
    {
        $query = Store::with(['division', 'district', 'upazila', 'union']);

        // Filter by division
        if ($request->filled('division_id')) {
            $query->byDivision($request->division_id);
        }

        // Filter by district
        if ($request->filled('district_id')) {
            $query->byDistrict($request->district_id);
        }

        // Filter by upazila
        if ($request->filled('upazila_id')) {
            $query->byUpazila($request->upazila_id);
        }

        // Filter by union
        if ($request->filled('union_id')) {
            $query->byUnion($request->union_id);
        }

        // Filter by city
        if ($request->filled('city')) {
            $query->byCity($request->city);
        }

        // Filter by area
        if ($request->filled('area')) {
            $query->byArea($request->area);
        }

        // Search by store name
        if ($request->filled('search')) {
            $query->where('name', 'LIKE', '%' . $request->search . '%');
        }

        // Get stores with pagination
        $stores = $query->latest()->paginate(15)->appends($request->query());

        // Get all divisions for filter dropdown
        $divisions = Division::orderBy('name_en')->get();
        
        // Get districts if division is selected
        $districts = [];
        if ($request->filled('division_id')) {
            $districts = District::where('division_id', $request->division_id)
                ->orderBy('name_en')
                ->get();
        }
        
        // Get upazilas if district is selected
        $upazilas = [];
        if ($request->filled('district_id')) {
            $upazilas = Upazila::where('district_id', $request->district_id)
                ->orderBy('name_en')
                ->get();
        }
        
        // Get unions if upazila is selected
        $unions = [];
        if ($request->filled('upazila_id')) {
            $unions = Union::where('upazila_id', $request->upazila_id)
                ->orderBy('name_en')
                ->get();
        }

        // Get all cities for filter dropdown
        $cities = Store::distinct()->pluck('city')->sort()->values();

        return view('backend.stores.index', compact(
            'stores', 
            'cities', 
            'divisions', 
            'districts', 
            'upazilas', 
            'unions'
        ));
    }

    /**
     * Show the form for creating a new store.
     */
    public function create()
    {
        $divisions = Division::orderBy('name_en')->get();
        return view('backend.stores.create', compact('divisions'));
    }

    /**
     * Store a newly created store in storage.
     */
    public function store(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'address' => 'required|string|max:500',
            'division_id' => 'required|exists:divisions,id',
            'district_id' => 'required|exists:districts,id',
            'upazila_id' => 'required|exists:upazilas,id',
            'union_id' => 'nullable|exists:unions,id',
            'city' => 'required|string|max:100',
            'area' => 'required|string|max:100',
            'latitude' => 'required|numeric|between:-90,90',
            'longitude' => 'required|numeric|between:-180,180',
            'phone' => 'required|string|max:20',
        ]);

        if ($validator->fails()) {
            return redirect()
                ->back()
                ->withErrors($validator)
                ->withInput();
        }

        try {
            Store::create($request->all());

            flash(translate('Store has been created successfully'))->success();
            return redirect()->route('admin.stores.index');
        } catch (\Exception $e) {
            flash(translate('Something went wrong'))->error();
            return redirect()->back()->withInput();
        }
    }

    /**
     * Display the specified store.
     */
    public function show($id)
    {
        $store = Store::with(['division', 'district', 'upazila', 'union'])->findOrFail($id);
        return view('backend.stores.show', compact('store'));
    }

    /**
     * Show the form for editing the specified store.
     */
    public function edit($id)
    {
        $store = Store::findOrFail($id);
        $divisions = Division::orderBy('name_en')->get();
        
        // Get districts based on selected division
        $districts = District::where('division_id', $store->division_id)
            ->orderBy('name_en')
            ->get();
            
        // Get upazilas based on selected district
        $upazilas = Upazila::where('district_id', $store->district_id)
            ->orderBy('name_en')
            ->get();
            
        // Get unions based on selected upazila
        $unions = Union::where('upazila_id', $store->upazila_id)
            ->orderBy('name_en')
            ->get();
        
        return view('backend.stores.edit', compact(
            'store', 
            'divisions', 
            'districts', 
            'upazilas', 
            'unions'
        ));
    }

    /**
     * Update the specified store in storage.
     */
    public function update(Request $request, $id)
    {
        $validator = Validator::make($request->all(), [
            'name' => 'required|string|max:255',
            'address' => 'required|string|max:500',
            'division_id' => 'required|exists:divisions,id',
            'district_id' => 'required|exists:districts,id',
            'upazila_id' => 'required|exists:upazilas,id',
            'union_id' => 'nullable|exists:unions,id',
            'city' => 'required|string|max:100',
            'area' => 'required|string|max:100',
            'latitude' => 'required|numeric|between:-90,90',
            'longitude' => 'required|numeric|between:-180,180',
            'phone' => 'required|string|max:20',
        ]);

        if ($validator->fails()) {
            return redirect()
                ->back()
                ->withErrors($validator)
                ->withInput();
        }

        try {
            $store = Store::findOrFail($id);
            $store->update($request->all());

            flash(translate('Store has been updated successfully'))->success();
            return redirect()->route('admin.stores.index');
        } catch (\Exception $e) {
            flash(translate('Something went wrong'))->error();
            return redirect()->back()->withInput();
        }
    }

    /**
     * Remove the specified store from storage.
     */
    public function destroy($id)
    {
        try {
            $store = Store::findOrFail($id);
            $storeName = $store->name;
            $store->delete();

            flash(translate('Store has been deleted successfully'))->success();
            return redirect()->route('admin.stores.index');
        } catch (\Exception $e) {
            flash(translate('Something went wrong'))->error();
            return redirect()->back();
        }
    }

    /**
     * Get districts by division (AJAX)
     */
    public function getDistrictsByDivision(Request $request)
    {
        try {
            $districts = District::where('division_id', $request->division_id)
                ->select('id', 'name_en', 'name_bn')
                ->orderBy('name_en')
                ->get();

            return response()->json([
                'success' => true,
                'data' => $districts
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error loading districts'
            ], 500);
        }
    }

    /**
     * Get upazilas by district (AJAX)
     */
    public function getUpazilasByDistrict(Request $request)
    {
        try {
            $upazilas = Upazila::where('district_id', $request->district_id)
                ->select('id', 'name_en', 'name_bn')
                ->orderBy('name_en')
                ->get();

            return response()->json([
                'success' => true,
                'data' => $upazilas
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error loading upazilas'
            ], 500);
        }
    }

    /**
     * Get unions by upazila (AJAX)
     */
    public function getUnionsByUpazila(Request $request)
    {
        try {
            $unions = Union::where('upazila_id', $request->upazila_id)
                ->select('id', 'name_en', 'name_bn')
                ->orderBy('name_en')
                ->get();

            return response()->json([
                'success' => true,
                'data' => $unions
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error loading unions'
            ], 500);
        }
    }

    /**
     * Get stores by city (AJAX)
     */
    public function storesByCity(Request $request)
    {
        try {
            $stores = Store::where('city', $request->city)
                ->with(['division', 'district', 'upazila', 'union'])
                ->select('id', 'name', 'area', 'address', 'division_id', 'district_id', 'upazila_id', 'union_id')
                ->orderBy('area')
                ->get();

            return response()->json([
                'success' => true,
                'data' => $stores
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error loading stores'
            ], 500);
        }
    }

    /**
     * Get stores by area (AJAX)
     */
    public function storesByArea(Request $request)
    {
        try {
            $query = Store::with(['division', 'district', 'upazila', 'union']);

            if ($request->has('city')) {
                $query->where('city', $request->city);
            }

            if ($request->has('area')) {
                $query->where('area', $request->area);
            }

            $stores = $query->select('id', 'name', 'address', 'phone', 'latitude', 'longitude', 'division_id', 'district_id', 'upazila_id', 'union_id')
                ->orderBy('name')
                ->get();

            return response()->json([
                'success' => true,
                'data' => $stores
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error loading stores'
            ], 500);
        }
    }

    /**
     * Get all cities (AJAX)
     */
    public function getCities()
    {
        try {
            $cities = Store::distinct()
                ->pluck('city')
                ->sort()
                ->values();

            return response()->json([
                'success' => true,
                'data' => $cities
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error loading cities'
            ], 500);
        }
    }

    /**
     * Get areas by city (AJAX)
     */
    public function getAreas(Request $request)
    {
        try {
            $areas = Store::where('city', $request->city)
                ->distinct()
                ->pluck('area')
                ->sort()
                ->values();

            return response()->json([
                'success' => true,
                'data' => $areas
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'message' => 'Error loading areas'
            ], 500);
        }
    }

    /**
     * Bulk delete stores
     */
    public function bulkDelete(Request $request)
    {
        try {
            if ($request->has('ids')) {
                Store::whereIn('id', $request->ids)->delete();
                
                flash(translate('Stores have been deleted successfully'))->success();
            } else {
                flash(translate('No stores selected'))->warning();
            }

            return redirect()->route('admin.stores.index');
        } catch (\Exception $e) {
            flash(translate('Something went wrong'))->error();
            return redirect()->back();
        }
    }

    /**
     * Export stores to CSV
     */
    public function export()
    {
        try {
            $stores = Store::with(['division', 'district', 'upazila', 'union'])
                ->orderBy('division_id')
                ->orderBy('district_id')
                ->orderBy('upazila_id')
                ->orderBy('union_id')
                ->orderBy('city')
                ->orderBy('area')
                ->get();
            
            $filename = "stores_" . date('Y-m-d_H-i-s') . ".csv";
            
            header('Content-Type: text/csv');
            header('Content-Disposition: attachment; filename="' . $filename . '"');
            
            $output = fopen('php://output', 'w');
            
            // Add CSV headers
            fputcsv($output, [
                'ID',
                'Store Name',
                'Address',
                'Division',
                'District',
                'Upazila',
                'Union',
                'City',
                'Area',
                'Phone',
                'Latitude',
                'Longitude',
                'Created At'
            ]);
            
            // Add data rows
            foreach ($stores as $store) {
                fputcsv($output, [
                    $store->id,
                    $store->name,
                    $store->address,
                    $store->division?->name_en ?? '',
                    $store->district?->name_en ?? '',
                    $store->upazila?->name_en ?? '',
                    $store->union?->name_en ?? '',
                    $store->city,
                    $store->area,
                    $store->phone,
                    $store->latitude,
                    $store->longitude,
                    $store->created_at->format('Y-m-d H:i:s')
                ]);
            }
            
            fclose($output);
            exit;
        } catch (\Exception $e) {
            flash(translate('Error exporting stores'))->error();
            return redirect()->back();
        }
    }
}