<?php

namespace App\Http\Controllers\Api\V2;

use App\Http\Controllers\Controller;
use App\Models\Store;
use Illuminate\Http\Request;

class StoreController extends Controller
{
    /**
     * Get stores with optional filtering by city and area.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function index(Request $request)
    {

        // Validate query parameters
        $request->validate([
            'city' => 'nullable|string|max:255',
            'area' => 'nullable|string|max:255',
        ]);

        $city = $request->query('city');
        $area = $request->query('area');

        // Build query with filters
        $stores = Store::query()
            ->byCity($city)
            ->byArea($area)
            ->orderBy('city')
            ->orderBy('area')
            ->get();

        return response()->json([
            'success' => true,
            'count' => $stores->count(),
            'data' => $stores,
        ]);
    }

      /**
     * Get list of all available cities
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function getCities()
    {
        $cities = Store::select('city')
            ->distinct()
            ->orderBy('city')
            ->pluck('city');

        return response()->json([
            'success' => true,
            'data' => $cities,
        ]);
    }

    /**
     * Get areas for a specific city
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\JsonResponse
     */
    public function getAreas(Request $request)
    {
        $request->validate([
            'city' => 'required|string|max:255',
        ]);

        $city = $request->query('city');

        $areas = Store::where('city', $city)
            ->select('area')
            ->distinct()
            ->orderBy('area')
            ->pluck('area');

        return response()->json([
            'success' => true,
            'city' => $city,
            'data' => $areas,
        ]);
    }

    
}