<?php

namespace App\Console\Commands;

use App\Models\Division;
use App\Models\District;
use App\Models\Upazila;
use App\Models\Union;
use App\Services\DictionaryTranslationService;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Symfony\Component\DomCrawler\Crawler;

class ScrapeBangladeshGeoData extends Command
{
    protected $signature = 'scrape:bangladesh-geo-data';
    protected $description = 'Scrape Bangladesh geographical data from bangladesh.gov.bd';
    protected $baseUrl = 'https://bangladesh.gov.bd';
    
    protected $translationService;
    
    public function __construct(DictionaryTranslationService $translationService)
    {
        parent::__construct();
        $this->translationService = $translationService;
    }
    
    public function handle()
    {
        $this->info('Starting to scrape Bangladesh geographical data...');
        
        // Clear existing data
        $this->clearExistingData();
        
        // Scrape divisions
        $divisions = $this->scrapeDivisions();
        $this->info("Found " . count($divisions) . " divisions");
        
        // Scrape districts for each division
        foreach ($divisions as $division) {
            $this->info("Processing division: {$division['name_en']} ({$division['name_bn']})");
            $divisionModel = Division::create([
                'id' => $division['id'],
                'name_en' => $division['name_en'],
                'name_bn' => $division['name_bn'],
            ]);
            
            $districts = $this->scrapeDistricts($division['id']);
            $this->info("Found " . count($districts) . " districts in {$division['name_en']}");
            
            // Scrape upazilas for each district
            foreach ($districts as $district) {
                $this->info("Processing district: {$district['name_en']} ({$district['name_bn']})");
                $districtModel = District::create([
                    'id' => $district['id'],
                    'division_id' => $divisionModel->id,
                    'name_en' => $district['name_en'],
                    'name_bn' => $district['name_bn'],
                ]);
                
                $upazilas = $this->scrapeUpazilas($district['id']);
                $this->info("Found " . count($upazilas) . " upazilas in {$district['name_en']}");
                
                // Scrape unions for each upazila
                foreach ($upazilas as $upazila) {
                    $this->info("Processing upazila: {$upazila['name_en']} ({$upazila['name_bn']})");
                    $upazilaModel = Upazila::create([
                        'id' => $upazila['id'],
                        'district_id' => $districtModel->id,
                        'name_en' => $upazila['name_en'],
                        'name_bn' => $upazila['name_bn'],
                    ]);
                    
                    $unions = $this->scrapeUnions($upazila['id']);
                    $this->info("Found " . count($unions) . " unions in {$upazila['name_en']}");
                    
                    // Save unions
                    foreach ($unions as $union) {
                        Union::create([
                            'id' => $union['id'],
                            'upazila_id' => $upazilaModel->id,
                            'name_en' => $union['name_en'],
                            'name_bn' => $union['name_bn'],
                            'url' => $union['url'] ?? null,
                        ]);
                    }
                }
            }
        }
        
        $this->info('Scraping completed successfully!');
        return Command::SUCCESS;
    }
    
    private function clearExistingData()
    {
        $this->info('Clearing existing data...');
        DB::statement('SET FOREIGN_KEY_CHECKS=0;');
        Union::truncate();
        Upazila::truncate();
        District::truncate();
        Division::truncate();
        DB::statement('SET FOREIGN_KEY_CHECKS=1;');
        $this->info('Existing data cleared.');
    }
    
    private function scrapeDivisions()
    {
        $this->info('Scraping divisions...');
        
        try {
            // Get the main page
            $response = Http::get($this->baseUrl . '/site/view/union-list/%20%E0%A6%87%E0%A6%89%E0%A6%BF%E0%A6%AF%E0%A6%BC%E0%A6%A8%E0%A6%B8%E0%A6%AE%E0%A6%B9');
            
            if (!$response->successful()) {
                $this->error('Failed to fetch the main page');
                return [];
            }
            
            $html = $response->body();
            $crawler = new Crawler($html);
            
            $divisions = [];
            
            // Extract divisions from the select dropdown
            $crawler->filter('#div-list option')->each(function (Crawler $node) use (&$divisions) {
                $value = $node->attr('value');
                $name = $node->text();
                
                if (!empty($value)) {
                    $divisions[] = [
                        'id' => $value,
                        'name_en' => $this->translationService->translateBanglaToEnglish($name),
                        'name_bn' => $name,
                    ];
                }
            });
            
            return $divisions;
        } catch (\Exception $e) {
            $this->error('Error scraping divisions: ' . $e->getMessage());
            Log::error('Scraping divisions error', ['message' => $e->getMessage()]);
            return [];
        }
    }
    
    private function scrapeDistricts($divisionId)
    {
        if (empty($divisionId)) {
            $this->error('Division ID is required');
            return [];
        }
        
        $this->info('Scraping districts for division ID: ' . $divisionId);
        
        try {
            // Make POST request to get districts
            $response = Http::asForm()->post($this->baseUrl . '/child.domains.bangla.php', [
                'parent' => $divisionId,
                'domain_type' => 'District'
            ]);
            
            if (!$response->successful()) {
                $this->error('Failed to fetch districts for division ID: ' . $divisionId);
                return [];
            }
            
            $html = $response->body();
            $crawler = new Crawler($html);
            
            $districts = [];
            
            // Extract districts from the response
            $crawler->filter('option')->each(function (Crawler $node) use (&$districts) {
                $value = $node->attr('value');
                $name = $node->text();
                
                if (!empty($value)) {
                    $districts[] = [
                        'id' => $value,
                        'name_en' => $this->translationService->translateBanglaToEnglish($name),
                        'name_bn' => $name,
                    ];
                }
            });
            
            return $districts;
        } catch (\Exception $e) {
            $this->error('Error scraping districts: ' . $e->getMessage());
            Log::error('Scraping districts error', ['message' => $e->getMessage()]);
            return [];
        }
    }
    
    private function scrapeUpazilas($divisionId)
    {
        if (empty($divisionId)) {
            $this->error('Division ID is required');
            return [];
        }
        
        $this->info('Scraping upazilas for district ID: ' . $divisionId);
        
        try {
            // Make POST request to get upazilas
            $response = Http::asForm()->post($this->baseUrl . '/child.domains.bangla.php', [
                'parent' => $divisionId,
                'domain_type' => 'Upazilla'
            ]);
            
            if (!$response->successful()) {
                $this->error('Failed to fetch upazilas for district ID: ' . $divisionId);
                return [];
            }
            
            $html = $response->body();
            $crawler = new Crawler($html);
            
            $upazilas = [];
            
            // Extract upazilas from the response
            $crawler->filter('option')->each(function (Crawler $node) use (&$upazilas) {
                $value = $node->attr('value');
                $name = $node->text();
                
                if (!empty($value)) {
                    $upazilas[] = [
                        'id' => $value,
                        'name_en' => $this->translationService->translateBanglaToEnglish($name),
                        'name_bn' => $name,
                    ];
                }
            });
            
            return $upazilas;
        } catch (\Exception $e) {
            $this->error('Error scraping upazilas: ' . $e->getMessage());
            Log::error('Scraping upazilas error', ['message' => $e->getMessage()]);
            return [];
        }
    }
    
    private function scrapeUnions($upazilaId)
    {
        if (empty($upazilaId)) {
            $this->error('Upazila ID is required');
            return [];
        }
        
        $this->info('Scraping unions for upazila ID: ' . $upazilaId);
        
        try {
            // Make POST request to get unions
            $response = Http::asForm()->post($this->baseUrl . '/child.domains.bangla.php', [
                'parent' => $upazilaId,
                'domain_type' => 'Union'
            ]);
            
            if (!$response->successful()) {
                $this->error('Failed to fetch unions for upazila ID: ' . $upazilaId);
                return [];
            }
            
            $html = $response->body();
            $crawler = new Crawler($html);
            
            $unions = [];
            
            // Extract unions from the response
            // For unions, the response format is different - it's a table with links
            $crawler->filter('.geotable tr td a')->each(function (Crawler $node) use (&$unions) {
                $url = $node->attr('href');
                $name = $node->text();
                
                if (!empty($url)) {
                    // Extract ID from URL
                    $id = $this->extractIdFromUrl($url);
                    
                    $unions[] = [
                        'id' => $id,
                        'name_en' => $this->translationService->translateBanglaToEnglish($name),
                        'name_bn' => $name,
                        'url' => $url,
                    ];
                }
            });
            
            return $unions;
        } catch (\Exception $e) {
            $this->error('Error scraping unions: ' . $e->getMessage());
            Log::error('Scraping unions error', ['message' => $e->getMessage()]);
            return [];
        }
    }
    
    private function extractIdFromUrl($url)
    {
        // Extract ID from URL like http://amlabaup.narsingdi.gov.bd
        // The pattern is: http://[name]up.[district_code].gov.bd
        if (preg_match('/http:\/\/([a-z]+)up\.([a-z]+)\.gov\.bd/', $url, $matches)) {
            return $matches[2]; // Return district code
        }
        
        return null;
    }
}